/*==============================================================================
PART 2C: Deposit Franchise Value Calculation
==============================================================================
Purpose:
This script calculates the estimated value of a bank's deposit franchise,
decomposed into insured and uninsured components. The calculation is based on
estimated deposit betas, interest rates, deposit costs, and deposit composition.

Input:
- $path_clean/call_reports_prepared.dta (Prepared bank-quarter panel from Part 1)
- $path_temp/deposit_betas${ext_suffix}.dta (Estimated deposit betas from Part 2A)
- $path_temp/deposit_costs${ext_suffix}.dta (Estimated deposit costs from Part 2B)
- $path_raw/tenyearrate${ir_suffix}.dta (FRED interest rate data)

Output:
- A Stata dataset in long panel form containing bank-period observations.
  Each observation identifies `rssdid`, `yq` (the end quarter of the period),
  and `period` (the name of the estimation period). It includes the estimated
  deposit franchise values (`DI_value`, `DU_value`, `DF_value`) and their
  changes from the dec2021 baseline for that specific period.
  Example structure (simplified):
     +-----------------------------------------------------------------+
     | rssdid       yq      period   DI_value   DU_value   DF_value |
     |-----------------------------------------------------------------|
  1. |   6329   2019q2   dec2021      150.00     80.00      230.00 |
  2. |   6329   2022q4   feb2023      120.00     60.00      180.00 |
  3. |   6329   2023q4   feb2024      140.00     70.00      210.00 |
  4. |   6330   2019q2   dec2021      100.00     50.00      150.00 |
  5. |   6330   2022q4   feb2023       90.00     45.00      135.00 |
     |-----------------------------------------------------------------|
  6. |   6330   2023q4   feb2024      110.00     55.00      165.00 |
     ......continued .....

Methodology:
1. Load the prepared bank panel and apply sample restrictions.
2. Merge in estimated deposit betas and costs (both in long format by period).
3. Merge in interest rate data (by yq).
4. Calculate the estimated value of insured (DI) and uninsured (DU) deposit
   franchises using the formula discussed in the paper.
5. Save the resulting long format dataset.

Last updated: July 25, 2025
==============================================================================*/

display "--- Starting Part 2C: Deposit Franchise Value Calculation ---" // Indicate the start of the script

clear all // Clear memory of any previous dataset
use "$path_clean/call_reports_prepared.dta", clear // Load the prepared bank-quarter panel data

// Define the end quarters for the estimation periods and filter the data
local dt_2021q4 = tq(2021q4) // dec2021 period end
local dt_2022q4 = tq(2022q4) // feb2023 period end
local dt_2023q4 = tq(2023q4) // feb2024 period end

// Keep only observations corresponding to the end quarters of the defined periods
keep if yq == `dt_2021q4' | yq == `dt_2022q4' | yq == `dt_2023q4'

// Create a string variable to identify the period for each observation
gen period = ""
replace period = "dec2021" if yq == `dt_2021q4'
replace period = "feb2023" if yq == `dt_2022q4'
replace period = "feb2024" if yq == `dt_2023q4'

sort rssdid period // Sort by bank and period for merging
//===============================================================================
// Step 1: Merge Long Format Data
//===============================================================================
// Purpose: Merge in estimated deposit betas and costs (both in long format by
// bank and period) and interest rate data (by quarter).
// Note: the ${ext_suffix} global is used to differentiate between different
// extensions in section 04, which reuse this code.

// Merge in estimated deposit betas from Part 2A (by rssdid and period)
merge 1:1 rssdid period using "$path_temp/deposit_betas${ext_suffix}.dta", keep(master match) nogen

// Merge in estimated deposit costs from Part 2B (by rssdid and period)
merge 1:1 rssdid period using "$path_temp/deposit_costs${ext_suffix}.dta", keep(master match) nogen

// Merge in 10-Year Treasury Rate
if "${ir_suffix}" == "" | "${ir_suffix}" == "." { // If no suffix or default suffix, use raw data
	merge m:1 yq using "$path_raw/tenyearrate${ir_suffix}.dta", keepusing(dgs10 f1*) keep(master match) nogen

}
else { // if extension, use data from the temporary directory (for extensions)
	merge m:1 yq using "$path_temp/tenyearrate${ir_suffix}.dta", keepusing(dgs10 f1*) keep(master match) nogen

}

sort rssdid period // Re-sort after merging


//===============================================================================
// Step 2: Calculate Deposit Franchise Values
//===============================================================================
// Purpose: Estimate the value of insured (DI) and uninsured (DU) deposit
// franchises using a single vectorized calculation that works across all periods.

// Create the appropriate interest rate variable based on the period
// The variable f1_dgs10 is constructed in fetch_fred_data.do to represent the
// 10-year Treasury rate at the start of the *next* quarter.
gen interest_rate = .
replace interest_rate = dgs10 if period == "dec2021" // For dec2021 period (end Q4 2021), use current dgs10 (as of 31 Dec 2021)
replace interest_rate = f1_dgs10 if period == "feb2023" // For feb2023 period (end Q4 2022), use f1_dgs10 (rate as of 26 Feb 2023)
replace interest_rate = f1_dgs10 if period == "feb2024" // For feb2024 period (end Q4 2023), use f1_dgs10 (rate as of 29 Feb 2024)

// Apply the uninsured deposits cap if the global ${uninsured_cap} is set below 1
if ${uninsured_cap} < 1 {
    display "Applying uninsured deposit cap at ${uninsured_cap}"
    replace uninsuredsh_domdep = ${uninsured_cap} if uninsuredsh_domdep > ${uninsured_cap}
}

// Calculate DI and DU values using the formula:
// Value = ((1 - beta) * InterestRate - Cost) / (InterestRate + runoff) * 100 * DepositShare * TotalDeposits
// Note: Costs (cost_ins, cost_unins) are in percentage points, so divide by 100.
// Deposit shares (1-uninsuredsh_domdep, uninsuredsh_domdep) are applied to total domestic deposits (domdep_assets * assets).
gen DI_value = ((1 - beta_ins) * interest_rate - (cost_ins / 100)) / (interest_rate + ${runoff}) * 100 * domdep_assets * (1 - uninsuredsh_domdep)
gen DU_value = ((1 - beta_unins) * interest_rate - (cost_unins / 100)) / (interest_rate + ${runoff}) * 100 * domdep_assets * uninsuredsh_domdep

//===============================================================================
// Step 3: Final Calculations
//===============================================================================
// Purpose: Calculate total deposit franchise value and truncated versions.

// Calculate total deposit franchise value as the sum of insured and uninsured components
gen DF_value = DI_value + DU_value

// Calculate truncated versions of DU_value and DF_value (at 0)
// Used in extensions
gen DU_value_trunc = max(DU_value, 0)
gen DI_value_trunc = max(DI_value, 0)
gen DF_value_trunc = max(DF_value, 0)

//===============================================================================
// Step 4: Save Long Format Data
//===============================================================================
// Purpose: Keep only the identifier variables and the newly created deposit
// franchise value variables and save the dataset to the temporary directory.

keep rssdid period D* // Keep rssdid, period, and all variables starting with 'D' (DI_value, DU_value, DF_value, etc.)

// Sort the final dataset
sort rssdid period

// Save the dataset
save "$path_temp/deposit_franchises${ext_suffix}.dta", replace

display "--- Deposit Franchise Value Calculation completed ---" // Indicate the completion of the script
